package mcfall.raytracer;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionListener;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.prefs.Preferences;

import javax.imageio.ImageIO;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.UIManager;
import javax.swing.UnsupportedLookAndFeelException;
import javax.swing.WindowConstants;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.filechooser.FileFilter;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPathExpressionException;

import net.miginfocom.swing.MigLayout;

import org.apache.log4j.FileAppender;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.xml.sax.SAXException;

public class TracerWindow extends JFrame implements AdvancedInfomation  {
	public static final String prefsKey = "LastDirectory";
	public int maxthreads = 2;
	
	private TracerWindow self = this;
	private JLabel imageLabel;
	private JPanel mainPanel;
	private JPanel optionsPanel;
	private JCheckBox debugOn;
	private JCheckBox shadowsEnabled;
	private JCheckBox reflectionEnabled;
	private JCheckBox transparencyEnabled;
	private JCheckBox boundingBoxEnabled;
	private JCheckBox extrudeEnabled;
	private JButton chooseFileButton;
	private JButton startButton;
	private JButton viewLogButton;
	private JButton advancedOptionsButton;
	private PixelInfoBox infoBox;
	private JTextField filename;
	private JTextField viewPlaneWidth;
	private JTextField viewPlaneHeight;
	private JTabbedPane tabPane; 
	private Preferences preferences;
	private ArrayList<RayTracerImageLabel> imageLabels = new ArrayList<RayTracerImageLabel>();
	DocumentListener filenameDocumentListener;
	
	/**
	 * The Scene object that represents the currently loaded scene file
	 */
	private Scene scene;
	
	public class TraceRecord {
		public String sceneFilename;
		public boolean debugEnabled;
		public boolean shadowsEnabled;
		public boolean reflectionEnabled;
		public boolean transparencyEnabled;
		public String debugLogFilename;
		public String viewPlaneWidth;
		public String viewPlaneHeight;
		protected boolean boundingBoxEnabled;
		protected boolean extrudeEnabled;		
	}
	
	private class FilenameChangeListener implements DocumentListener {	
		public void changedUpdate(DocumentEvent arg0) {
			enableStartOnValidFilename(arg0);
		}			
		public void insertUpdate(DocumentEvent arg0) {
			enableStartOnValidFilename(arg0);
		}		
		public void removeUpdate(DocumentEvent arg0) {
			enableStartOnValidFilename(arg0);
		}		
		public void enableStartOnValidFilename (DocumentEvent arg0) {
			Document doc = arg0.getDocument();
			int length = doc.getLength();
			try {
				String filename = doc.getText(0, length);
				File file = new File (filename);										
				if (file.exists()) {
					try {
						scene = new Scene (TracerWindow.this.filename.getText());
						viewPlaneWidth.setText(String.valueOf(scene.getCamera().getPlaneWidth()));
						viewPlaneHeight.setText(String.valueOf(scene.getCamera().getPlaneHeight()));
						startButton.setEnabled (true);
					}
					catch (Exception ex) {
						JOptionPane.showMessageDialog(TracerWindow.this, "An error occurred processing " + file + ".  The error message was: " + ex.getMessage(), "Error reading file", JOptionPane.ERROR_MESSAGE);
						startButton.setEnabled(false);
					}
				}
			}
			catch (BadLocationException ex) {
				//  This should never happen
				ex.printStackTrace();
			}
		}
	}
	
	private HashMap<String, TraceRecord> tabInfo;
	
	public TracerWindow () {
	
		super ("Ray Tracer");
		try {
			UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
		} catch (ClassNotFoundException e2) {
			// TODO Auto-generated catch block
			e2.printStackTrace();
		} catch (InstantiationException e2) {
			// TODO Auto-generated catch block
			e2.printStackTrace();
		} catch (IllegalAccessException e2) {
			// TODO Auto-generated catch block
			e2.printStackTrace();
		} catch (UnsupportedLookAndFeelException e2) {
			// TODO Auto-generated catch block
			e2.printStackTrace();
		}
		this.setJMenuBar(createJMenuBar());
		
		tabInfo = new HashMap<String, TraceRecord> ();
		
		setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
		preferences = Preferences.userNodeForPackage(RayTracer.class);
		setLayout (new MigLayout ());
		
		new JMenuBar ();
		mainPanel = new JPanel (new MigLayout ());
		debugOn = new JCheckBox ("Enable debugging", false);
		chooseFileButton = new JButton ("Open ...");
		startButton = new JButton ("Start");
		startButton.setEnabled(false);
		filename = new JTextField (25);		
		imageLabel = new JLabel ("Rendered Image");
		shadowsEnabled = new JCheckBox ("Enable shadows", true);
		reflectionEnabled = new JCheckBox ("Enable reflection", true);
		transparencyEnabled = new JCheckBox("Enable transparency",true);
		boundingBoxEnabled = new JCheckBox("Enable bounding boxes",true);
		extrudeEnabled = new JCheckBox("Enable extrudes",true);
		viewPlaneWidth = new JTextField (4);
		viewPlaneHeight = new JTextField (4);
		
		advancedOptionsButton = new JButton("Advanced");
		advancedOptionsButton.addActionListener(new OpenAdvancedOptionsListener());
		tabPane = new JTabbedPane ();		
		imageLabel.setVerticalTextPosition(SwingConstants.TOP);
		imageLabel.setHorizontalAlignment(SwingConstants.CENTER);
		viewLogButton = new JButton ("View debug log");
		viewLogButton.setEnabled(false);
		
		optionsPanel = new JPanel (new MigLayout ());
		JPanel renderingOptions = new JPanel (new MigLayout ());
		renderingOptions.setBorder(BorderFactory.createTitledBorder("Rendering Options"));
		renderingOptions.add(shadowsEnabled);
		renderingOptions.add(reflectionEnabled, "wrap");
		renderingOptions.add (transparencyEnabled, "wrap");
		renderingOptions.add (new JLabel ("Width"), "span 2, split 4");
		renderingOptions.add (viewPlaneWidth);
		renderingOptions.add (new JLabel ("Height"));
		renderingOptions.add (viewPlaneHeight);
		
		JPanel performanceOptions = new JPanel (new MigLayout ());
		performanceOptions.setBorder(BorderFactory.createTitledBorder("Performance"));
		performanceOptions.add (boundingBoxEnabled);
		performanceOptions.add (extrudeEnabled, "wrap");
		performanceOptions.add (debugOn);
		
		JPanel fileSelectionPanel = new JPanel (new MigLayout ());
		fileSelectionPanel.setBorder(BorderFactory.createTitledBorder("File to render"));
		fileSelectionPanel.add (new JLabel("Filename: "));
		fileSelectionPanel.add (filename, "wrap");
		fileSelectionPanel.add (chooseFileButton, "span 2, align center");
		
		infoBox = new PixelInfoBox();
		
		mainPanel.add (fileSelectionPanel, "aligny top");
		mainPanel.add (renderingOptions, "aligny top");
		mainPanel.add (performanceOptions, "aligny top, wrap");		
		mainPanel.add (startButton, "span 3, split 2, align center");
		mainPanel.add (viewLogButton, "align center, wrap");
		mainPanel.add (infoBox, "growx, wrap");		
				
		add (tabPane, "wrap, width 640:540:, height 480:480:, growy, growx");
		add (mainPanel, "wrap");
		
		
		tabPane.addChangeListener( new ChangeListener () {
			public void stateChanged(ChangeEvent e) {
				String title = tabPane.getTitleAt(tabPane.getSelectedIndex());
				TraceRecord info = tabInfo.get(title);
				shadowsEnabled.setSelected(info.shadowsEnabled);
				debugOn.setSelected(info.debugEnabled);
				
				// Temporarily remove the listener that checks for a valid file and
				// parses it; if we don't do this, then the value in the file will 
				// be used for the view plane width/height even if the user has overriden
				// them.
				filename.getDocument().removeDocumentListener(filenameDocumentListener);
				filename.setText(info.sceneFilename);
				filename.getDocument().addDocumentListener(filenameDocumentListener);
				
				viewPlaneWidth.setText(info.viewPlaneWidth);
				viewPlaneHeight.setText(info.viewPlaneHeight);
				viewLogButton.setEnabled(info.debugEnabled);
			}			
		});
		
		chooseFileButton.addActionListener( openFileListener(preferences));
		
		debugOn.addActionListener( new ActionListener () {
			public void actionPerformed(ActionEvent e) {
				if (debugOn.isSelected()) {
					Logger.getRootLogger().setLevel(Level.DEBUG);
				}
				else {
					Logger.getRootLogger().setLevel(Level.ERROR);
				}							
			}			
		});
		
		startButton.addActionListener(new ActionListener () {
			private void reportInvalidValue (String type, String value) {
				JOptionPane.showMessageDialog(TracerWindow.this, "The value " + value + " is invalid for the view plane " + type.toLowerCase(), "Invalid value", JOptionPane.ERROR_MESSAGE);
			}
			
			public void actionPerformed(ActionEvent e) {
				RayTracer tracer;

				int viewPlaneWidthValue = -1;
				int viewPlaneHeightValue = -1;
				
				try {
					viewPlaneWidthValue = Integer.valueOf(viewPlaneWidth.getText()); 
				}
				catch (NumberFormatException badWidth) {
					reportInvalidValue ("Width", viewPlaneWidth.getText());
					return;
				}
				
				try {
					viewPlaneHeightValue = Integer.valueOf(viewPlaneHeight.getText()); 
				}
				catch (NumberFormatException badHeight) {
					reportInvalidValue ("Height", viewPlaneHeight.getText());
					return;
				}
				
				scene.getCamera().setPlaneSize(new Dimension (viewPlaneWidthValue, viewPlaneHeightValue));
				
				TraceRecord record = new TraceRecord ();
				record.debugEnabled = debugOn.isSelected();
				record.sceneFilename = filename.getText();
				record.shadowsEnabled = shadowsEnabled.isSelected();
				record.reflectionEnabled = reflectionEnabled.isSelected();
				record.transparencyEnabled = transparencyEnabled.isSelected();
				record.boundingBoxEnabled = boundingBoxEnabled.isSelected();
				record.extrudeEnabled = extrudeEnabled.isSelected();
				record.viewPlaneHeight = viewPlaneHeight.getText();
				record.viewPlaneWidth = viewPlaneWidth.getText();
				
				if (debugOn.isSelected()) {
					try {
						File debugFile = File.createTempFile("TraceDebug", ".txt");		
						debugFile.deleteOnExit();
						FileAppender appender = (FileAppender) Logger.getRootLogger().getAppender("file");
						if (appender != null) {	
							appender.setFile(debugFile.getAbsolutePath());
							appender.activateOptions();														
							record.debugLogFilename = debugFile.getAbsolutePath();
						}
					}
					catch (IOException ex) {
						JOptionPane.showMessageDialog(TracerWindow.this, "An error occurred attempting to create the debug file: " + ex.getMessage() + "; debugging will be turned off.", "Error creating debug file", JOptionPane.ERROR_MESSAGE);
						record.debugEnabled = false;
					}
				}
				else {
					record.debugLogFilename = null;
				}

				tracer = new RayTracer (scene);
				tracer.setShadowsEnabled(shadowsEnabled.isSelected());
				tracer.setReflectionEnabled(reflectionEnabled.isSelected());
				tracer.setTransparencyEnabled(transparencyEnabled.isSelected());
				tracer.setBoundingBoxEnabled(boundingBoxEnabled.isSelected());
				tracer.setExtrudeEnabled(extrudeEnabled.isSelected());

				String tabLabel = record.sceneFilename.substring(record.sceneFilename.lastIndexOf(File.separator)+1);
				if (tabInfo.containsKey(tabLabel)) {
					int i = 2;					
					while (tabInfo.containsKey(tabLabel + " - " + i)) {
						i++;
					}
					tabLabel = tabLabel + " - " + i;
				}										
				tabInfo.put(tabLabel, record);

				final RayTracerImageLabel newLabel = new RayTracerImageLabel(tracer,SwingConstants.CENTER,getMaxThreads());
				imageLabels.add(newLabel);//make the "labels" easily accessable

				newLabel.setVerticalTextPosition(SwingConstants.TOP);					
				newLabel.setHorizontalTextPosition(SwingConstants.CENTER);
				newLabel.startTrace(getGridSubsX(), getGridSubsX());

				newLabel.addMouseMotionListener(new MouseMotionListener () {
					public void mouseDragged(MouseEvent e) {}

					public void mouseMoved(MouseEvent e) {
						Point location = e.getPoint();
						JLabel label = (JLabel) e.getSource();							
						Icon icon = label.getIcon();
						int left = (label.getWidth()-icon.getIconWidth())/2;
						int top = (label.getHeight() - icon.getIconHeight())/2;
						int x = location.x - left;
						int y = location.y - top;

						if (x >= 0 && y >= 0 && x < icon.getIconWidth() && y < icon.getIconHeight()) {

							int color = newLabel.getImage().getRGB(x, y);
							int red = (color & 0x00FF0000) >> 16;
						int green = (color & 0x0000FF00) >> 8;
						int blue = (color & 0x000000FF);
						y = (int) (icon.getIconHeight()-y);
						infoBox.setInfo(x, y, red, green, blue);
						}
					}					
				});
				final JScrollPane scrollPane = new JScrollPane();
				//TODO why doesn't the pane resize in *nix?
				scrollPane.setPreferredSize(new Dimension(800,600));
				scrollPane.getViewport().add(newLabel);					
				tabPane.addTab(tabLabel,scrollPane);
				tabPane.setSelectedComponent(scrollPane);
				TracerWindow.this.pack ();
				TracerWindow.this.repaint();


			}			
		});

		this.filenameDocumentListener = new FilenameChangeListener ();
		filename.getDocument().addDocumentListener (filenameDocumentListener);
						
		viewLogButton.addActionListener( new ActionListener () {
			public void actionPerformed (ActionEvent e) {
				//TODO is it ironic that the debug window will very possibly crash java?  The debug.txt file can be MASSIVE; need to find a way to increase the java heap at runtime
				String title = tabPane.getTitleAt(tabPane.getSelectedIndex());
				TraceRecord record = tabInfo.get(title);
				String filename = record.debugLogFilename;
				DebugWindow debugWindow = new DebugWindow (TracerWindow.this, title, filename);
				debugWindow.setVisible(true);								
			}
		});
		pack ();
		setVisible (true);
	}
	private ActionListener openFileListener(final Preferences preferences) {
		return new ActionListener () {
			public void actionPerformed(ActionEvent e) {
				JFileChooser chooser = new JFileChooser ();
				String lastDirectory = preferences.get(prefsKey, null); 
				if (lastDirectory != null) {
					chooser.setCurrentDirectory(new File(lastDirectory));
				}
				
				chooser.setFileFilter(new FileFilter () {
					@Override
					public boolean accept(File f) {
						String filename = f.getAbsolutePath();
						if (f.isDirectory()) return true;
						int lastDot = filename.lastIndexOf(".");
						if (lastDot == -1) {
							return false;
						}
						String extension = filename.substring(lastDot);
						return (extension.equalsIgnoreCase(".XML"));
					}

					@Override
					public String getDescription() {
						return "XML Files";
					}
					
				});
				
				int result = chooser.showOpenDialog(null);
				if (result == JFileChooser.APPROVE_OPTION) {				
					filename.setText(chooser.getSelectedFile().getAbsolutePath());
					preferences.put(prefsKey, chooser.getCurrentDirectory().getAbsolutePath());
				}
			}
			
		};
	}
	class GenericFileFilter extends FileFilter{
		String extension;
		String description; 
		public GenericFileFilter(String extension, String description)
		{
			this.extension = extension;
			this.description = description;
		}
		@Override
		public boolean accept(File f) {
			String filename = f.getAbsolutePath();
			if (f.isDirectory()) return true;
			int lastDot = filename.lastIndexOf(".");
			if (lastDot == -1) {
				return false;
			}
			String extension = filename.substring(lastDot);
			return (extension.equalsIgnoreCase(this.extension));
		}
		@Override
		public String getDescription() {
			return this.description;
		}							
	};
	public JMenuBar createJMenuBar() 
	{
		JMenuBar menuBar = new JMenuBar();
		JMenu fileMenu = new JMenu("File");
		fileMenu.setMnemonic(KeyEvent.VK_F);
		menuBar.add(fileMenu);
		JMenuItem saveMenuItem = new JMenuItem("Save",KeyEvent.VK_S);
		fileMenu.add(saveMenuItem);
		saveMenuItem.addActionListener(
				new ActionListener() 
				{
					public void actionPerformed(ActionEvent e) {
						JFileChooser chooser = new JFileChooser ();
						chooser.setAcceptAllFileFilterUsed(false);						
						chooser.addChoosableFileFilter(new GenericFileFilter(".gif",".gif file with LZW (gif) lossless compression (low quality)"));
						chooser.addChoosableFileFilter(new GenericFileFilter(".jpg",".jpg image file with lossy compression"));
						chooser.addChoosableFileFilter(new GenericFileFilter(".tiff",".tiff file with LZW (gif) lossless compression"));
						chooser.addChoosableFileFilter(new GenericFileFilter(".png",".png image file with the DEFLATE lossless compression (best quality)"));
						int result = chooser.showSaveDialog(null);
						if (result == JFileChooser.APPROVE_OPTION) {
							int i = tabPane.getSelectedIndex();
							RayTracerImageLabel currentLabel = imageLabels.get(i);
							BufferedImage currentImage = currentLabel.getImage();
							File file = chooser.getSelectedFile();
							try {
								String extension = file.getName().substring(file.getName().lastIndexOf('.')+1);
								ImageIO.write(currentImage,extension,file);
							} catch (IOException e1) {
								// TODO Auto-generated catch block
								e1.printStackTrace();
							}
						}
					}
				}
		);
		
		return menuBar;
	}
	/* (non-Javadoc)
	 * @see mcfall.raytracer.AdvancedInfomation#setGridSubsX(int)
	 */
	public void setGridSubsX(int gridSubsX) {
		preferences.put("gridSubsX", ""+gridSubsX);
	}
	/* (non-Javadoc)
	 * @see mcfall.raytracer.AdvancedInfomation#getGridSubsX()
	 */
	public int getGridSubsX() {
		return Integer.parseInt(preferences.get("gridSubsX", "16")); //get our saved preference
	}
	/* (non-Javadoc)
	 * @see mcfall.raytracer.AdvancedInfomation#setGridSubsY(int)
	 */
	public void setGridSubsY(int gridSubsY) {
		preferences.put("gridSubsY", ""+gridSubsY);
	}
	/* (non-Javadoc)
	 * @see mcfall.raytracer.AdvancedInfomation#getGridSubsY()
	 */
	public int getGridSubsY() {
		return Integer.parseInt(preferences.get("gridSubsY", "16"));
	}
	/* (non-Javadoc)
	 * @see mcfall.raytracer.AdvancedInfomation#setMaxThreads(int)
	 */
	public void setMaxThreads(int maxThreads) {
		preferences.put("maxThreads", ""+maxThreads);
	}
	/* (non-Javadoc)
	 * @see mcfall.raytracer.AdvancedInfomation#getMaxThreads()
	 */
	public int getMaxThreads() {
		return Integer.parseInt(preferences.get("maxThreads", "1"));
	}
	
	public void setMaxRecursions(int maxRecursions) {
		preferences.put("maxRecursions", ""+maxRecursions);
	}
	
	public int getMaxRecursions() {
		return Integer.parseInt(preferences.get("maxRecursions", "10"));
	}
	
	public class OpenAdvancedOptionsListener implements ActionListener{
		public void actionPerformed(ActionEvent e) {
			new AdvancedOptions(self);			
		}
		
	}
}
